--
-- Enhanced fillType categories
--
-- Author: Wopster
-- Description: adds missing fill types to fill type categories.
-- Icon:
-- Name: Enhanced filltype categories
-- Hide: no
--
-- Copyright (c) Wopster, 2024

---@type string directory of the mod.
local modDirectory = g_currentModDirectory or ""
---@type string name of the mod.
local modName = g_currentModName or "unknown"

local function addFillTypeToCategory(fillTypeName, categoryName)
    local fillType = g_fillTypeManager:getFillTypeByName(fillTypeName:upper())
    local categoryIndex = g_fillTypeManager.nameToCategoryIndex[categoryName:upper()]
    if fillType ~= nil and categoryIndex ~= nil then
        if not g_fillTypeManager:addFillTypeToCategory(fillType.index, categoryIndex) then
            log(("Error: failed to add %s to %s"):format(fillTypeName, categoryName))
        end
    end
end

local function registerFillTypes()
    addFillTypeToCategory("lime", "manureSpreader")
    addFillTypeToCategory("lime", "spreader")
    addFillTypeToCategory("lime", "augerWagon")
    addFillTypeToCategory("water", "slurryTank")
end

local effectsToMerge = {
    ["FERTILIZER"] = {
        into = "MANURE_SPREADER",
        fallbackFillTypeStr = "MANURE",
        fallbackEffectMeshIndex = 1
    }
}

local function mergeEffects()
    for from, intoData in pairs(effectsToMerge) do
        local intoEffects = g_motionPathEffectManager.effectsByType[intoData.into:upper()]

        if intoEffects ~= nil then
            local fromEffects = g_motionPathEffectManager.effectsByType[from:upper()]

            for _, intoEffect in pairs(intoEffects) do

                if #intoEffect.effectMaterials == 0 and intoData.fallbackFillTypeStr ~= nil then
                    local node = intoEffect.effectMeshes[intoData.fallbackEffectMeshIndex or 1].node

                    local effectMaterial = {
                        node = node,
                        materialId = getMaterial(node, 0),
                        parent = intoEffect,
                        lod = {},
                        fillTypes = {
                            g_fillTypeManager:getFillTypeIndexByName(intoData.fallbackFillTypeStr)
                        },
                        fruitTypes = nil,
                        growthStates = nil,
                        customShaderMaps = {},
                        customShaderParameters = {},
                    }

                    table.insert(intoEffect.effectMaterials, effectMaterial)
                end

                for _, fromEffect in pairs(fromEffects) do
                    intoEffect.customShaderVariation = fromEffect.customShaderVariation

                    for _, effectMesh in pairs(fromEffect.effectMeshes) do
                        table.insert(intoEffect.effectMeshes, effectMesh)
                    end

                    for _, effectMaterial in pairs(fromEffect.effectMaterials) do
                        table.insert(intoEffect.effectMaterials, effectMaterial)
                    end
                end
            end
        else
            Logging.error("Enhanced fillType categories is missing the effect type %s in your map, please check for mod conflicts!", intoData.into)
        end
    end
end

---Init the mod.
local function init()
    FillTypeManager.loadMapData = Utils.appendedFunction(FillTypeManager.loadMapData, registerFillTypes)
    FSBaseMission.onFinishedLoading = Utils.prependedFunction(FSBaseMission.onFinishedLoading, mergeEffects)
end

init()
