-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
THEventManager = {}
local THEventManager_mt = THUtils.createClass(THEventManager, THObject)
THEventManager.NETWORK_ID_BITS = 16
THEventManager.MESSAGE = {
    NETWORK_ID_MISMATCH = "Network id mismatch: %s",
    NETWORK_ID_LIMIT_REACHED = "Maximum network objects exceeded: %s"
}
function THEventManager.new(customMt, objectMt)
    customMt = customMt or THEventManager_mt
    local self = THObject.new(g_server ~= nil, g_client ~= nil, customMt, objectMt)
    if self ~= nil then
        self.modName = thModName
        self.modPath = thModPath
        self.taskManager = g_asyncTaskManager
        self.delayedLoadTasks = {}
        self.isEnabled = true
        self.isRegisteredInStream = false
        self.areActionEventsRegistered = false
        self.isPreInitFinished = false
        self.isPostInitFinished = false
        self.networkObjects = {}
        self.networkObjectToId = {}
        self.networkIdToObject = {}
        self.eventListeners = {}
        THUtils.setFunctionHook("XMLManager", "createSchemas", false, false, self, THEventManager.inj_createSchemas)
        THUtils.setFunctionHook("XMLManager", "initSchemas", false, false, self, THEventManager.inj_initSchemas)
        THUtils.setFunctionHook("Mission00", "setMissionInfo", false, false, self, THEventManager.inj_setMissionInfo)
        THUtils.setFunctionHook("Mission00", "loadEnvironment", false, false, self, THEventManager.inj_loadEnvironment)
        THUtils.setFunctionHook("FSBaseMission", "initTerrain", false, false, self, THEventManager.inj_initTerrain)
        THUtils.setFunctionHook("FSBaseMission", "loadMapFinished", false, false, self, THEventManager.inj_loadMapFinished)
        THUtils.setFunctionHook("Player", "readStream", false, false, self, THEventManager.inj_initialReadStream)
        THUtils.setFunctionHook("Player", "writeStream", false, false, self, THEventManager.inj_initialWriteStream)
        THUtils.setFunctionHook("FSCareerMissionInfo", "saveToXMLFile", false, false, self, THEventManager.inj_saveToXMLFile)
        return self
    end
end
function THEventManager.onDelete(self)
    THEventManager:superClass().onDelete(self)
    self.isEnabled = false
end
function THEventManager.onUpdate(self, dt)
    THEventManager:superClass().onUpdate(self, dt)
    if self:getIsEnabled() then
        if not self.areActionEventsRegistered then
            self:raiseEvent("onRegisterActionEvents")
            self.areActionEventsRegistered = true
        end
        self:raiseEvent("onUpdate", dt)
    end
end
function THEventManager.onUpdateTick(self, dt)
    THEventManager:superClass().onUpdateTick(self, dt)
    if self:getIsEnabled() then
        self:raiseEvent("onUpdateTick", dt)
    end
end
function THEventManager.onReadStream(self, streamId, connection)
    THEventManager:superClass().onReadStream(self, streamId, connection)
    local isServerEnabled = streamReadBool(streamId)
    if isServerEnabled then
        if connection:getIsServer() then
            local numNetworkObjects = streamReadUIntN(streamId, THEventManager.NETWORK_ID_BITS)
            THUtils.call(function()
                THUtils.clearTable(self.networkIdToObject)
                THUtils.clearTable(self.networkObjectToId)
            end)
            if numNetworkObjects > 0 then
                for objectIndex = 1, numNetworkObjects do
                    local networkObject = self.networkObjects[objectIndex]
                    local networkId = streamReadUIntN(streamId, THEventManager.NETWORK_ID_BITS)
                    if networkObject ~= nil then
                        self.networkObjectToId[networkObject] = networkId
                        if networkId > 0 then
                            self.networkIdToObject[networkId] = networkObject
                        end
                    end
                end
            end
        end
        self:raiseEvent("onReadStream", streamId, connection)
    else
        self.isEnabled = false
    end
end
function THEventManager.onWriteStream(self, streamId, connection)
    THEventManager:superClass().onWriteStream(self, streamId, connection)
    local isEnabled = self:getIsEnabled()
    if streamWriteBool(streamId, isEnabled == true) then
        if not connection:getIsServer() then
            local numNetworkObjects = #self.networkObjects
            streamWriteUIntN(streamId, numNetworkObjects, THEventManager.NETWORK_ID_BITS)
            if numNetworkObjects > 0 then
                for objectIndex = 1, numNetworkObjects do
                    local networkObject = self.networkObjects[objectIndex]
                    local networkId = self.networkObjectToId[networkObject] or 0
                    streamWriteUIntN(streamId, networkId, THEventManager.NETWORK_ID_BITS)
                end
            end
        end
        self:raiseEvent("onWriteStream", streamId, connection)
    end
end
function THEventManager.onReadUpdateStream(self, streamId, timestamp, connection)
    THEventManager:superClass().onReadUpdateStream(self, streamId, timestamp, connection)
    local isServerEnabled = streamReadBool(streamId)
    if isServerEnabled then
        self:raiseEvent("onReadUpdateStream", streamId, timestamp, connection)
    else
        self.isEnabled = false
    end
end
function THEventManager.onWriteUpdateStream(self, streamId, connection, dirtyMask)
    THEventManager:superClass().onWriteUpdateStream(self, streamId, connection, dirtyMask)
    local isEnabled = self:getIsEnabled()
    if streamWriteBool(streamId, isEnabled == true) then
        self:raiseEvent("onWriteUpdateStream", streamId, connection, dirtyMask)
    end
end
function THEventManager.getIsEnabled(self)
    return self.isEnabled == true
end
function THEventManager.addEventListener(self, target, modName)
    modName = modName or self.modName
    local modEnv = THUtils.getModEnvironment(modName)
    if THUtils.argIsValid(type(target) == "table", "target", target)
        and THUtils.argIsValid(modEnv ~= nil, "modName", modName)
        and THUtils.assert(target ~= self, true, "Cannot add event manager to listeners table")
    then
        if self:getIsEnabled() then
            local isListenerFound = false
            for _, otherListenerData in pairs(self.eventListeners) do
                if otherListenerData.target == target then
                    isListenerFound = true
                    break
                end
            end
            if not isListenerFound then
                local listenerData = {
                    target = target,
                    modName = modName,
                    modEnv = modEnv
                }
                table.insert(self.eventListeners, listenerData)
            end
        end
    end
end
function THEventManager.removeEventListener(self, target)
    local numListeners = #self.eventListeners
    if target ~= nil and numListeners > 0 then
        local listenerIndex = 1
        local listenerData = nil
        while true do
            listenerData = self.eventListeners[listenerIndex]
            if listenerData == nil then
                break
            end
            if target == listenerData.target then
                table.remove(self.eventListeners, listenerIndex)
            else
                listenerIndex = listenerIndex + 1
            end
        end
    end
end
function THEventManager.raiseEvent(self, eventName, ...)
    if self:getIsEnabled() then
        local numListeners = #self.eventListeners
        if numListeners > 0 then
            for listenerIndex = 1, numListeners do
                local listenerData = self.eventListeners[listenerIndex]
                if listenerData.target ~= nil and type(listenerData.target[eventName]) == "function" then
                    local callbackFunc = listenerData.target[eventName]
                    if listenerData.modName ~= self.modName then
                        setfenv(callbackFunc, listenerData.modEnv)
                    end
                    callbackFunc(listenerData.target, ...)
                end
            end
        end
    end
end
function THEventManager.addProtectedTask(self, taskFunc)
    if self:getIsEnabled() then
        self.taskManager:addTask(function()
            THUtils.pcall(taskFunc)
        end)
    end
end
function THEventManager.addProtectedSubTask(self, taskFunc)
    if self:getIsEnabled() then
        self.taskManager:addSubtask(function()
            THUtils.pcall(taskFunc)
        end)
    end
end
function THEventManager.addDelayedLoadTask(self, taskFunc)
    if THUtils.argIsValid(type(taskFunc) == "function", "taskFunc", taskFunc)
        and self:getIsEnabled()
    then
        table.insert(self.delayedLoadTasks, taskFunc)
        return true
    end
    return false
end
function THEventManager.getNetworkId(self, target)
    if target ~= nil then
        local networkId = self.networkObjectToId[target]
        if networkId ~= nil and networkId > 0 then
            if self.networkIdToObject[networkId] == target then
                return networkId
            end
            self.networkObjectToId[target] = 0
            THUtils.errorMsg(true, THEventManager.MESSAGE.NETWORK_ID_MISMATCH, networkId)
        end
    end
    return 0
end
function THEventManager.getTargetByNetworkId(self, networkId)
    if networkId ~= nil and networkId > 0 then
        local target = self.networkIdToObject[networkId]
        if target ~= nil then
            if self.networkObjectToId[target] ~= networkId then
                self.networkObjectToId[target] = networkId
                THUtils.errorMsg(true, THEventManager.MESSAGE.NETWORK_ID_MISMATCH, networkId)
            end
            return target
        end
    end
end
function THEventManager.createNetworkId(self, target)
    if THUtils.argIsValid(type(target) == "table", "target", target) then
        local numNetworkObjects = #self.networkObjects
        local networkIndex = 0
        for objectIndex = 1, numNetworkObjects do
            if self.networkObjects[objectIndex] == target then
                networkIndex = objectIndex
                break
            end
        end
        if networkIndex > 0 then
            local networkId = self:getNetworkId(target)
            if networkId > 0 then
                return networkId
            end
            THUtils.errorMsg(true, "Could not create network id")
        else
            local nextId = numNetworkObjects + 1
            local maxId = (2 ^ THEventManager.NETWORK_ID_BITS) - 1
            if nextId <= maxId then
                self.networkObjects[nextId] = target
                self.networkIdToObject[nextId] = target
                self.networkObjectToId[target] = nextId
                return nextId
            end
            THUtils.errorMsg(true, THEventManager.MESSAGE.NETWORK_ID_LIMIT_REACHED, maxId)
        end
    end
    return 0
end
function THEventManager.inj_createSchemas(self, superFunc, ...)
    if self:getIsEnabled() then
        THUtils.call(function()
            if not self.isPreInitFinished then
                self:raiseEvent("onPreInit")
                self.isPreInitFinished = true
            end
        end)
    end
    return superFunc(...)
end
function THEventManager.inj_initSchemas(self, superFunc, ...)
    local function appendFunc(...)
        if self:getIsEnabled() then
            THUtils.call(function()
                self:addProtectedSubTask(function()
                    local numTasks = #self.delayedLoadTasks
                    if numTasks > 0 then
                        for taskIndex = 1, numTasks do
                            THUtils.call(self.delayedLoadTasks[taskIndex])
                            self.delayedLoadTasks[taskIndex] = nil
                        end
                    end
                    if not self.isPostInitFinished then
                        self:raiseEvent("onPostInit")
                        self.isPostInitFinished = true
                    end
                end)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(...))
end
function THEventManager.inj_setMissionInfo(self, superFunc, mission, missionInfo, missionDynamicInfo, ...)
    local function appendFunc(...)
        if self:getIsEnabled() then
            THUtils.call(function()
                self:addProtectedTask(function()
                    self:raiseEvent("onSetMissionInfo", mission, missionInfo, missionDynamicInfo)
                end)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(mission, missionInfo, missionDynamicInfo, ...))
end
function THEventManager.inj_loadEnvironment(self, superFunc, mission, ...)
    local function appendFunc(...)
        if self:getIsEnabled() then
            THUtils.call(function()
                self:raiseEvent("onLoadEnvironment", mission.environment, mission)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(mission, ...))
end
function THEventManager.inj_initTerrain(self, superFunc, mission, terrainId, filename, ...)
    local function appendFunc(...)
        if self:getIsEnabled() then
            THUtils.call(function()
                self:raiseEvent("onInitTerrain", mission, terrainId, filename)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(mission, terrainId, filename, ...))
end
function THEventManager.inj_loadMapFinished(self, superFunc, mission, ...)
    local function appendFunc(...)
        if self:getIsEnabled() then
            THUtils.call(function()
                if not mission.cancelLoading then
                    self:raiseEvent("onLoadMapFinished", mission)
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(mission, ...))
end
function THEventManager.inj_initialReadStream(self, superFunc, player, streamId, connection, ...)
    local function appendFunc(...)
        if connection:getIsServer() then
            local isServerEnabled = streamReadBool(streamId)
            if streamReadBool(streamId) then
                if isServerEnabled then
                    local parent = self.parent
                    local updaterId = NetworkUtil.readNodeObjectId(streamId)
                    parent:readStream(streamId, connection)
                    g_client:finishRegisterObject(parent, updaterId)
                end
            end
            if not isServerEnabled then
                self.isEnabled = false
                THUtils.errorMsg(false, "Client THEventManager disabled by server")
            end
            self.isRegisteredInStream = true
        end
        return ...
    end
    return appendFunc(superFunc(player, streamId, connection, ...))
end
function THEventManager.inj_initialWriteStream(self, superFunc, player, streamId, connection, ...)
    local function appendFunc(...)
        if not connection:getIsServer() then
            local isEnabled = streamWriteBool(streamId, self.isEnabled == true)
            if streamWriteBool(streamId, self.isRegisteredInStream ~= true) then
                if isEnabled then
                    local parent = self.parent
                    local updaterId = NetworkUtil.getObjectId(parent)
                    NetworkUtil.writeNodeObjectId(streamId, updaterId)
                    parent:writeStream(streamId, connection)
                    g_server:registerObjectInStream(connection, parent)
                end
                self.isRegisteredInStream = true
            end
        end
        return ...
    end
    return appendFunc(superFunc(player, streamId, connection, ...))
end
function THEventManager.inj_saveToXMLFile(self, superFunc, missionInfo, ...)
    local function appendFunc(...)
        if self:getIsEnabled() and self.parent.isServer then
            THUtils.call(function()
                self:raiseEvent("onSaveToSavegame")
            end)
        end
        return ...
    end
    return appendFunc(superFunc(missionInfo, ...))
end
THUtils.call(function()
    local self = THEventManager.new()
    if self ~= nil then
        _G.g_thEventManager = self
    end
end)