PalletSpawnMode = {}

local modName = g_currentModName or "FS25_PalletSpawnStore"

PalletSpawnMode.MODE_STORE       = 0
PalletSpawnMode.MODE_SPAWN       = 1
PalletSpawnMode.MODE_SELL        = 2
PalletSpawnMode.MODE_DISTRIBUTE  = 3

function PalletSpawnMode._applySpawnDisabled(productionPoint, fillTypeId, disabled)
    productionPoint.psm_spawnDisabled = productionPoint.psm_spawnDisabled or {}

    if disabled then
        productionPoint.psm_spawnDisabled[fillTypeId] = true
    else
        productionPoint.psm_spawnDisabled[fillTypeId] = nil
    end

    PalletSpawnMode.updatePalletSpawnForFillType(productionPoint, fillTypeId)
end

function PalletSpawnMode.setSpawnDisabled(productionPoint, fillTypeId, disabled, noEventSend)
    PalletSpawnMode._applySpawnDisabled(productionPoint, fillTypeId, disabled)

    if noEventSend ~= true then
        PalletSpawnModeEvent.sendEvent(productionPoint, fillTypeId, disabled)
    end
end

function PalletSpawnMode.updatePalletSpawnForFillType(productionPoint, fillTypeId)
    if productionPoint.palletSpawner == nil then
        return
    end

    productionPoint.psm_allPalletFillTypes = productionPoint.psm_allPalletFillTypes or {}

    if productionPoint.outputFillTypeIdsToPallets == nil then
        productionPoint.outputFillTypeIdsToPallets = {}
    end

    local allMap   = productionPoint.psm_allPalletFillTypes
    local curMap   = productionPoint.outputFillTypeIdsToPallets
    local disabled = productionPoint.psm_spawnDisabled ~= nil
                     and productionPoint.psm_spawnDisabled[fillTypeId] == true

    if allMap[fillTypeId] == nil then
        allMap[fillTypeId] = curMap[fillTypeId]
    end

    if disabled then
        curMap[fillTypeId] = nil
    else
        local cfg = allMap[fillTypeId]
        if cfg ~= nil then
            curMap[fillTypeId] = cfg
        end
    end
end

function PalletSpawnMode.getExtendedMode(productionPoint, fillTypeId)
    if productionPoint.getOutputDistributionMode == nil then
        return PalletSpawnMode.MODE_STORE
    end

    local baseMode = productionPoint:getOutputDistributionMode(fillTypeId)
    local spawnDisabled = productionPoint.psm_spawnDisabled ~= nil
        and productionPoint.psm_spawnDisabled[fillTypeId] == true

    if baseMode == ProductionPoint.OUTPUT_MODE.KEEP then
        if spawnDisabled then
            return PalletSpawnMode.MODE_STORE
        else
            return PalletSpawnMode.MODE_SPAWN
        end

    elseif baseMode == ProductionPoint.OUTPUT_MODE.DIRECT_SELL then
        return PalletSpawnMode.MODE_SELL

    elseif baseMode == ProductionPoint.OUTPUT_MODE.AUTO_DELIVER then
        return PalletSpawnMode.MODE_DISTRIBUTE
    end

    return PalletSpawnMode.MODE_STORE
end

function PalletSpawnMode.applyExtendedMode(productionPoint, fillTypeId, extMode, noEventSend)
    local baseMode = ProductionPoint.OUTPUT_MODE.KEEP
    local spawnDisabled = true

    if extMode == PalletSpawnMode.MODE_STORE then
        baseMode      = ProductionPoint.OUTPUT_MODE.KEEP
        spawnDisabled = true

    elseif extMode == PalletSpawnMode.MODE_SPAWN then
        baseMode      = ProductionPoint.OUTPUT_MODE.KEEP
        spawnDisabled = false

    elseif extMode == PalletSpawnMode.MODE_SELL then
        baseMode      = ProductionPoint.OUTPUT_MODE.DIRECT_SELL
        spawnDisabled = true

    elseif extMode == PalletSpawnMode.MODE_DISTRIBUTE then
        baseMode      = ProductionPoint.OUTPUT_MODE.AUTO_DELIVER
        spawnDisabled = true
    end

    productionPoint:setOutputDistributionMode(fillTypeId, baseMode, noEventSend)

    PalletSpawnMode.setSpawnDisabled(productionPoint, fillTypeId, spawnDisabled, noEventSend)
end


function PalletSpawnMode.cycleExtendedMode(productionPoint, fillTypeId)
    local curMode = PalletSpawnMode.getExtendedMode(productionPoint, fillTypeId)
    local nextMode = (curMode + 1) % 4

    PalletSpawnMode.applyExtendedMode(productionPoint, fillTypeId, nextMode, false)
end

PalletSpawnModeEvent = {}
local PalletSpawnModeEvent_mt = Class(PalletSpawnModeEvent, Event)

InitEventClass(PalletSpawnModeEvent, "PalletSpawnModeEvent")

function PalletSpawnModeEvent.emptyNew()
    return Event.new(PalletSpawnModeEvent_mt)
end

function PalletSpawnModeEvent.new(productionPoint, outputFillTypeId, spawnDisabled)
    local self = PalletSpawnModeEvent.emptyNew()
    self.productionPoint = productionPoint
    self.outputFillTypeId = outputFillTypeId
    self.spawnDisabled = spawnDisabled
    return self
end

function PalletSpawnModeEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.productionPoint)
    streamWriteUIntN(streamId, self.outputFillTypeId, FillTypeManager.SEND_NUM_BITS)
    streamWriteBool(streamId, self.spawnDisabled)
end

function PalletSpawnModeEvent:readStream(streamId, connection)
    self.productionPoint  = NetworkUtil.readNodeObject(streamId)
    self.outputFillTypeId = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)
    self.spawnDisabled    = streamReadBool(streamId)
    self:run(connection)
end

function PalletSpawnModeEvent:run(connection)
    if not connection:getIsServer() then
        g_server:broadcastEvent(self, false, connection)
    end

    if self.productionPoint ~= nil then
        PalletSpawnMode._applySpawnDisabled(
            self.productionPoint,
            self.outputFillTypeId,
            self.spawnDisabled
        )
    end
end


function PalletSpawnModeEvent.sendEvent(productionPoint, outputFillTypeId, spawnDisabled, noEventSend)
    if noEventSend == true then
        return
    end

    if g_server ~= nil then
        g_server:broadcastEvent(PalletSpawnModeEvent.new(productionPoint, outputFillTypeId, spawnDisabled))
    else
        g_client:getServerConnection():sendEvent(
            PalletSpawnModeEvent.new(productionPoint, outputFillTypeId, spawnDisabled)
        )
    end
end

local function overwritePopulateCell(self, superFunc, list, section, index, cell)
    if superFunc ~= nil then
        superFunc(self, list, section, index, cell)
    end

    if list ~= self.productsList then
        return
    end

    local _, productionPoint = self:getSelectedProduction()
    if productionPoint == nil or productionPoint.sortedProductions == nil then
        return
    end

    local production = productionPoint.sortedProductions[index]
    if production == nil then
        return
    end

    local fillTypeId = production.primaryProductFillType
    if fillTypeId == nil or fillTypeId == FillType.UNKNOWN then
        return
    end

    local mode = PalletSpawnMode.getExtendedMode(productionPoint, fillTypeId)
    local text

    if mode == PalletSpawnMode.MODE_STORE then
        text = g_i18n:getText("psm_prod_output_store")      -- Store (no pallets)
    elseif mode == PalletSpawnMode.MODE_SPAWN then
        text = g_i18n:getText("psm_prod_output_spawn")      -- Spawn / Store pallets
    elseif mode == PalletSpawnMode.MODE_SELL then
        text = g_i18n:getText("psm_prod_output_sell")       -- Sell
    elseif mode == PalletSpawnMode.MODE_DISTRIBUTE then
        text = g_i18n:getText("psm_prod_output_distribute") -- Distribute
    else
        text = ""
    end

    local activityElement = cell:getAttribute("activity")
    if activityElement ~= nil then
        activityElement:setText(text)
    end
end

local function overwriteOnButtonToggleOutputMode(self, superFunc)
    if not g_currentMission:getHasPlayerPermission("manageProductions") then
        InfoDialog.show(
            g_i18n:getText("shop_messageNoPermissionGeneral"),
            nil, nil,
            DialogElement.TYPE_WARNING
        )
        return
    end

    local production, productionPoint = self:getSelectedProduction()
    if production == nil or productionPoint == nil then
        return
    end

    local fillTypeId = production.primaryProductFillType
    if fillTypeId == nil or fillTypeId == FillType.UNKNOWN then
        return
    end

    PalletSpawnMode.cycleExtendedMode(productionPoint, fillTypeId)

    self.productsList:reloadData()
end

function PalletSpawnMode.saveCustom()
    psmDebug("Info: Saving Pallet Spawn Mode")
    if g_currentMission == nil or g_currentMission.productionChainManager == nil then
        return
    end

    local savegameFolderPath = g_currentMission.missionInfo.savegameDirectory
    local path = savegameFolderPath .. "/palletSpawnStore.xml"
    local key = "palletSpawnStore"

    local xmlFile = XMLFile.create(key, path, key)
    if xmlFile == nil then
        return
    end

    local index = 0

    for _, productionPoint in pairs(g_currentMission.productionChainManager.productionPoints) do
        local owningPlaceable = productionPoint.owningPlaceable
        local uniqueId = owningPlaceable and owningPlaceable.uniqueId

        if uniqueId ~= nil and productionPoint.psm_spawnDisabled ~= nil then
            for fillTypeId, disabled in pairs(productionPoint.psm_spawnDisabled) do
                if disabled then
                    local fillTypeName = g_fillTypeManager:getFillTypeNameByIndex(fillTypeId)
                    if fillTypeName ~= nil then
                        local subKey = string.format(".production(%d)", index)
                        xmlFile:setString(key .. subKey .. "#placeableUniqueId", uniqueId)
                        xmlFile:setString(key .. subKey .. "#fillType",          fillTypeName)
                        index = index + 1
                    end
                end
            end
        end
    end

    xmlFile:save()
    xmlFile:delete()
end


function PalletSpawnMode.loadCustom()
    psmDebug("Info: Loading Pallet Spawn Mode")
    if g_currentMission == nil or g_currentMission.productionChainManager == nil then
        return
    end

    local savegameFolderPath = g_currentMission.missionInfo.savegameDirectory
    local path = savegameFolderPath .. "/palletSpawnStore.xml"
    local key  = "palletSpawnStore"

    -- No file yet: first load of a brand-new savegame
    if not fileExists(path) then
        for _, productionPoint in pairs(g_currentMission.productionChainManager.productionPoints) do
            PalletSpawnMode.applyDefaultStoreMode(productionPoint)
        end
        return
    end

    local xmlFile = XMLFile.load(key, path, key)
    if xmlFile == nil then
        return
    end

    xmlFile:iterate(key .. ".production", function(_, entryKey)
        local uniqueId     = xmlFile:getString(entryKey .. "#placeableUniqueId")
        local fillTypeName = xmlFile:getString(entryKey .. "#fillType")

        if uniqueId ~= nil and fillTypeName ~= nil then
            local fillTypeId = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)
            if fillTypeId ~= nil then
                local productionPoint = PalletSpawnMode.getProductionPointsByUniqueId(uniqueId)
                if productionPoint then
                    PalletSpawnMode.setSpawnDisabled(productionPoint, fillTypeId, true, true)
                end
            end
        end
    end)

    xmlFile:delete()
end

function PalletSpawnMode.getProductionPointsByUniqueId(wantedUniqueId)
    if wantedUniqueId == nil then
        return nil
    end

    local wanted = tostring(wantedUniqueId)

    for _, productionPoint in pairs(g_currentMission.productionChainManager.productionPoints) do
        local owningPlaceable = productionPoint.owningPlaceable
        local uniqueId = owningPlaceable and owningPlaceable.uniqueId
        if uniqueId ~= nil and tostring(uniqueId) == wanted then
            return productionPoint
        end
    end

    return nil
end

function PalletSpawnMode.writeProductionStream(self, superFunc, streamId, connection)
    superFunc(self, streamId, connection)

    if not connection:getIsServer() then
        local numDisabled = 0

        if self.psm_spawnDisabled ~= nil then
            for _, disabled in pairs(self.psm_spawnDisabled) do
                if disabled then
                    numDisabled = numDisabled + 1
                end
            end
        end

        streamWriteUInt8(streamId, numDisabled)

        if numDisabled > 0 then
            for fillTypeId, disabled in pairs(self.psm_spawnDisabled) do
                if disabled then
                    streamWriteUIntN(streamId, fillTypeId, FillTypeManager.SEND_NUM_BITS)
                end
            end
        end
    end
end

function PalletSpawnMode.readProductionStream(self, superFunc, streamId, connection)
    superFunc(self, streamId, connection)

    if connection:getIsServer() then
        local numDisabled = streamReadUInt8(streamId)

        for _ = 1, numDisabled do
            local fillTypeId = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)
            PalletSpawnMode.setSpawnDisabled(self, fillTypeId, true, true)
        end
    end
end

function PalletSpawnMode.onSaveToXMLFile(missionInfo, superFunc, xmlFile, key)
    superFunc(missionInfo, xmlFile, key)

    if g_server ~= nil and g_currentMission ~= nil and g_currentMission.missionInfo == missionInfo then
        PalletSpawnMode.saveCustom()
    end
end

function PalletSpawnMode:loadMap(name)
    if InGameMenuProductionFrame ~= nil then
        InGameMenuProductionFrame.populateCellForItemInSection =
            Utils.overwrittenFunction(
                InGameMenuProductionFrame.populateCellForItemInSection,
                overwritePopulateCell
            )

        InGameMenuProductionFrame.onButtonToggleOutputMode =
            Utils.overwrittenFunction(
                InGameMenuProductionFrame.onButtonToggleOutputMode,
                overwriteOnButtonToggleOutputMode
            )
    end

    if ProductionPoint ~= nil then
        ProductionPoint.writeStream =
            Utils.overwrittenFunction(ProductionPoint.writeStream, PalletSpawnMode.writeProductionStream)
        ProductionPoint.readStream =
            Utils.overwrittenFunction(ProductionPoint.readStream, PalletSpawnMode.readProductionStream)
    end

    if FSCareerMissionInfo ~= nil then
        FSCareerMissionInfo.saveToXMLFile =
            Utils.overwrittenFunction(FSCareerMissionInfo.saveToXMLFile, PalletSpawnMode.onSaveToXMLFile)
    end
end

function PalletSpawnMode.applyDefaultStoreMode(productionPoint)
    if productionPoint == nil
        or productionPoint.outputFillTypeIds == nil then
        return
    end

    for _, fillTypeId in ipairs(productionPoint.outputFillTypeIds) do
        productionPoint:setOutputDistributionMode(fillTypeId,
            ProductionPoint.OUTPUT_MODE.KEEP,
            true
        )

        PalletSpawnMode.setSpawnDisabled(productionPoint,
            fillTypeId,
            true,
            true
        )
    end
end

function PalletSpawnMode:deleteMap() end

function PalletSpawnMode:keyEvent(unicode, sym, modifier, isDown) end

function PalletSpawnMode:mouseEvent(posX, posY, isDown, isUp, button) end

function PalletSpawnMode:update(dt)
    if g_server ~= nil and not self.psmHasLoaded then
        local mission = g_currentMission

        if mission ~= nil
            and mission.productionChainManager ~= nil
            and mission.missionInfo ~= nil then

            local pcm = mission.productionChainManager
            local points = pcm.productionPoints

            if points ~= nil then
                -- Check if there's at least one production point
                local hasAny = false
                for _ in pairs(points) do
                    hasAny = true
                    break
                end

                if hasAny then
                    self.psmHasLoaded = true
                    PalletSpawnMode.loadCustom()
                end
            end
        end
    end
end

function PalletSpawnMode:draw() end

addModEventListener(PalletSpawnMode)


---Simple debug function that does stuff with different types of vars.  
function psmDebug(data)
  local debug = false
  if debug then
    if data ~= nil then
      -- Run if data is table
      if type(data) == "table" then
        print("** PSM Debug - Table **")
        DebugUtil.printTableRecursively(data, "  tabledata : ", 0, 1)
      -- Run if data is boolean
      elseif type(data) == "boolean" then
        if data == true then 
          print("** PSM Debug - Boolean ** : true")
        else 
          print("** PSM Debug - Boolean ** : false")
        end
      -- Run if data is number
      elseif type(data) == "number" then
        print("** PSM Debug - Number ** : " .. data)
      -- Run if data is string
      elseif type(data) == "string" then
        print("** PSM Debug - String ** : " .. data)
      -- Run if data is function
      elseif type(data) == "function" then
        print("** PSM Debug - Function ** : ")
        print(data)
      -- Run if data is thread
      elseif type(data) == "thread" then
        print("** PSM Debug - Thread ** : ")
        print(data)
      -- Run if nothing else
      else
        print("** PSM Debug ** : " .. data)
      end
    else
      print("** PSM Debug ** : nil")
    end
  end
end